<?php
/**
 * کلاس مدیریت دیتابیس
 * Database Management Class
 */

class Database {
    private static $instance = null;
    private $connection = null;
    private $lastQuery = null;
    private $queryCount = 0;

    /**
     * Constructor - اتصال به دیتابیس
     */
    private function __construct() {
        try {
            $dsn = sprintf(
                "mysql:host=%s;dbname=%s;charset=%s",
                DB_HOST,
                DB_NAME,
                DB_CHARSET
            );

            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];

            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);

        } catch (PDOException $e) {
            $this->logError('Database Connection Failed: ' . $e->getMessage());
            die('خطا در اتصال به پایگاه داده. لطفاً با مدیر سیستم تماس بگیرید.');
        }
    }

    /**
     * Singleton Pattern
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * دریافت اتصال PDO
     */
    public function getConnection() {
        return $this->connection;
    }

    /**
     * اجرای کوئری SELECT
     */
    public function query($sql, $params = []) {
        try {
            $this->lastQuery = $sql;
            $this->queryCount++;

            if (LOG_QUERIES) {
                $this->logQuery($sql, $params);
            }

            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;

        } catch (PDOException $e) {
            $this->logError('Query Failed: ' . $e->getMessage() . ' | SQL: ' . $sql);
            throw new Exception('خطا در اجرای کوئری دیتابیس');
        }
    }

    /**
     * دریافت یک رکورد
     */
    public function fetchOne($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetch();
    }

    /**
     * دریافت همه رکوردها
     */
    public function fetchAll($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchAll();
    }

    /**
     * دریافت یک مقدار
     */
    public function fetchColumn($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchColumn();
    }

    /**
     * INSERT
     */
    public function insert($table, $data) {
        $columns = array_keys($data);
        $values = array_values($data);

        $placeholders = array_fill(0, count($columns), '?');

        $sql = sprintf(
            "INSERT INTO %s (%s) VALUES (%s)",
            $table,
            implode(', ', $columns),
            implode(', ', $placeholders)
        );

        $this->query($sql, $values);
        return $this->connection->lastInsertId();
    }

    /**
     * UPDATE
     */
    public function update($table, $data, $where, $whereParams = []) {
        $set = [];
        $values = [];

        foreach ($data as $column => $value) {
            $set[] = "$column = ?";
            $values[] = $value;
        }

        $sql = sprintf(
            "UPDATE %s SET %s WHERE %s",
            $table,
            implode(', ', $set),
            $where
        );

        $values = array_merge($values, $whereParams);
        $stmt = $this->query($sql, $values);
        return $stmt->rowCount();
    }

    /**
     * DELETE
     */
    public function delete($table, $where, $whereParams = []) {
        $sql = sprintf("DELETE FROM %s WHERE %s", $table, $where);
        $stmt = $this->query($sql, $whereParams);
        return $stmt->rowCount();
    }

    /**
     * شروع Transaction
     */
    public function beginTransaction() {
        return $this->connection->beginTransaction();
    }

    /**
     * Commit
     */
    public function commit() {
        return $this->connection->commit();
    }

    /**
     * Rollback
     */
    public function rollback() {
        return $this->connection->rollBack();
    }

    /**
     * آخرین ID درج شده
     */
    public function lastInsertId() {
        return $this->connection->lastInsertId();
    }

    /**
     * تعداد رکوردها
     */
    public function count($table, $where = '1=1', $params = []) {
        $sql = "SELECT COUNT(*) FROM $table WHERE $where";
        return (int) $this->fetchColumn($sql, $params);
    }

    /**
     * وجود رکورد
     */
    public function exists($table, $where, $params = []) {
        return $this->count($table, $where, $params) > 0;
    }

    /**
     * اجرای Stored Procedure
     */
    public function callProcedure($procName, $params = []) {
        try {
            $placeholders = array_fill(0, count($params), '?');
            $sql = "CALL $procName(" . implode(',', $placeholders) . ")";

            return $this->query($sql, $params);

        } catch (PDOException $e) {
            $this->logError('Procedure Failed: ' . $e->getMessage());
            throw new Exception('خطا در اجرای رویه ذخیره شده');
        }
    }

    /**
     * صفحه‌بندی
     */
    public function paginate($sql, $params, $page = 1, $perPage = ITEMS_PER_PAGE) {
        // شمارش کل رکوردها
        $countSql = "SELECT COUNT(*) FROM ($sql) AS count_table";
        $total = (int) $this->fetchColumn($countSql, $params);

        // محاسبه صفحه‌بندی
        $totalPages = ceil($total / $perPage);
        $page = max(1, min($page, $totalPages));
        $offset = ($page - 1) * $perPage;

        // اضافه کردن LIMIT به کوئری
        $sql .= " LIMIT $perPage OFFSET $offset";
        $data = $this->fetchAll($sql, $params);

        return [
            'data' => $data,
            'pagination' => [
                'total' => $total,
                'per_page' => $perPage,
                'current_page' => $page,
                'total_pages' => $totalPages,
                'has_prev' => $page > 1,
                'has_next' => $page < $totalPages
            ]
        ];
    }

    /**
     * لاگ کوئری
     */
    private function logQuery($sql, $params) {
        $log = date('Y-m-d H:i:s') . " | Query: $sql";
        if (!empty($params)) {
            $log .= " | Params: " . json_encode($params, JSON_UNESCAPED_UNICODE);
        }
        error_log($log . PHP_EOL, 3, LOG_DIR . 'queries.log');
    }

    /**
     * لاگ خطا
     */
    private function logError($message) {
        if (LOG_ERRORS) {
            $log = date('Y-m-d H:i:s') . " | Error: $message";
            error_log($log . PHP_EOL, 3, LOG_DIR . 'errors.log');
        }
    }

    /**
     * دریافت آمار کوئری‌ها
     */
    public function getQueryCount() {
        return $this->queryCount;
    }

    /**
     * دریافت آخرین کوئری
     */
    public function getLastQuery() {
        return $this->lastQuery;
    }

    /**
     * بستن اتصال
     */
    public function close() {
        $this->connection = null;
    }

    /**
     * جلوگیری از Clone
     */
    private function __clone() {}

    /**
     * جلوگیری از Unserialize
     */
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}

?>