<?php
/**
 * Database Class
 * کلاس مدیریت دیتابیس با PDO
 */

class Database {
    private static $instance = null;
    private $pdo;
    private $stmt;
    private $error;

    /**
     * Constructor - Private برای Singleton
     */
    private function __construct() {
        $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=' . DB_CHARSET;
        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
        ];

        try {
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            $this->error = $e->getMessage();
            if (DEBUG_MODE) {
                die('Database Connection Error: ' . $this->error);
            } else {
                die('خطا در اتصال به دیتابیس');
            }
        }
    }

    /**
     * Singleton Instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Prepare SQL Query
     */
    public function query($sql) {
        $this->stmt = $this->pdo->prepare($sql);
        return $this;
    }

    /**
     * Bind Values
     */
    public function bind($param, $value, $type = null) {
        if (is_null($type)) {
            switch (true) {
                case is_int($value):
                    $type = PDO::PARAM_INT;
                    break;
                case is_bool($value):
                    $type = PDO::PARAM_BOOL;
                    break;
                case is_null($value):
                    $type = PDO::PARAM_NULL;
                    break;
                default:
                    $type = PDO::PARAM_STR;
            }
        }
        $this->stmt->bindValue($param, $value, $type);
        return $this;
    }

    /**
     * Execute Query
     */
    public function execute() {
        try {
            return $this->stmt->execute();
        } catch (PDOException $e) {
            $this->error = $e->getMessage();
            if (DEBUG_MODE) {
                die('Query Error: ' . $this->error);
            }
            return false;
        }
    }

    /**
     * Fetch All Results
     */
    public function fetchAll() {
        $this->execute();
        return $this->stmt->fetchAll();
    }

    /**
     * Fetch Single Result
     */
    public function fetch() {
        $this->execute();
        return $this->stmt->fetch();
    }

    /**
     * Fetch Single Column
     */
    public function fetchColumn() {
        $this->execute();
        return $this->stmt->fetchColumn();
    }

    /**
     * Get Row Count
     */
    public function rowCount() {
        return $this->stmt->rowCount();
    }

    /**
     * Get Last Insert ID
     */
    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }

    /**
     * Begin Transaction
     */
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }

    /**
     * Commit Transaction
     */
    public function commit() {
        return $this->pdo->commit();
    }

    /**
     * Rollback Transaction
     */
    public function rollback() {
        return $this->pdo->rollback();
    }

    /**
     * Get Error
     */
    public function getError() {
        return $this->error;
    }

    /**
     * Helper: Select
     */
    public function select($table, $columns = '*', $where = '', $params = []) {
        $sql = "SELECT {$columns} FROM {$table}";
        if ($where) {
            $sql .= " WHERE {$where}";
        }

        $this->query($sql);

        if (!empty($params)) {
            foreach ($params as $key => $value) {
                $this->bind($key, $value);
            }
        }

        return $this->fetchAll();
    }

    /**
     * Helper: Insert
     */
    public function insert($table, $data) {
        $columns = implode(', ', array_keys($data));
        $placeholders = ':' . implode(', :', array_keys($data));

        $sql = "INSERT INTO {$table} ({$columns}) VALUES ({$placeholders})";
        $this->query($sql);

        foreach ($data as $key => $value) {
            $this->bind(":{$key}", $value);
        }

        if ($this->execute()) {
            return $this->lastInsertId();
        }
        return false;
    }

    /**
     * Helper: Update
     */
    public function update($table, $data, $where, $params = []) {
        $set = [];
        foreach (array_keys($data) as $key) {
            $set[] = "{$key} = :{$key}";
        }
        $set = implode(', ', $set);

        $sql = "UPDATE {$table} SET {$set} WHERE {$where}";
        $this->query($sql);

        foreach ($data as $key => $value) {
            $this->bind(":{$key}", $value);
        }

        foreach ($params as $key => $value) {
            $this->bind($key, $value);
        }

        return $this->execute();
    }

    /**
     * Helper: Delete
     */
    public function delete($table, $where, $params = []) {
        $sql = "DELETE FROM {$table} WHERE {$where}";
        $this->query($sql);

        foreach ($params as $key => $value) {
            $this->bind($key, $value);
        }

        return $this->execute();
    }

    /**
     * Helper: Count
     */
    public function count($table, $where = '', $params = []) {
        $sql = "SELECT COUNT(*) FROM {$table}";
        if ($where) {
            $sql .= " WHERE {$where}";
        }

        $this->query($sql);

        foreach ($params as $key => $value) {
            $this->bind($key, $value);
        }

        return $this->fetchColumn();
    }

    /**
     * جلوگیری از Clone
     */
    private function __clone() {}

    /**
     * جلوگیری از Unserialize
     */
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}
