<?php
/**
 * Auth Class
 * کلاس احراز هویت و مدیریت دسترسی ادمین
 */

class Auth {
    private $db;
    private $admin_id;
    private $admin_data;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->checkSession();
    }

    /**
     * بررسی Session
     */
    private function checkSession() {
        if (isset($_SESSION['admin_id'])) {
            $this->admin_id = $_SESSION['admin_id'];
            $this->loadAdminData();
        }
    }

    /**
     * ورود ادمین
     */
    public function login($email, $password, $remember = false) {
        // بررسی تعداد تلاش‌های ناموفق
        if ($this->isLoginBlocked($email)) {
            return [
                'success' => false,
                'message' => 'به دلیل تلاش‌های ناموفق زیاد، ورود شما موقتاً مسدود شده است.'
            ];
        }

        // یافتن کاربر
        $user = $this->db->query("
            SELECT u.*, a.id as admin_id, a.role, a.permissions
            FROM users u
            INNER JOIN admins a ON u.id = a.user_id
            WHERE u.email = :email AND u.status = 'active'
            LIMIT 1
        ")
        ->bind(':email', $email)
        ->fetch();

        if (!$user) {
            $this->logFailedAttempt($email);
            return [
                'success' => false,
                'message' => 'ایمیل یا رمز عبور اشتباه است.'
            ];
        }

        // بررسی رمز عبور
        if (!password_verify($password, $user['password_hash'])) {
            $this->logFailedAttempt($email);
            return [
                'success' => false,
                'message' => 'ایمیل یا رمز عبور اشتباه است.'
            ];
        }

        // ایجاد Session
        $this->createSession($user['admin_id'], $user);

        // ثبت لاگ ورود موفق
        $this->logActivity($user['admin_id'], 'login', 'admin', $user['admin_id']);

        // بروزرسانی آخرین لاگین
        $this->db->update('users', [
            'last_login' => date('Y-m-d H:i:s'),
            'last_ip' => $this->getClientIP(),
            'last_user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'total_login_count' => $user['total_login_count'] + 1
        ], 'id = :id', [':id' => $user['id']]);

        return [
            'success' => true,
            'message' => 'ورود موفق'
        ];
    }

    /**
     * ایجاد Session
     */
    private function createSession($admin_id, $user) {
        $_SESSION['admin_id'] = $admin_id;
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['role'] = $user['role'];
        $_SESSION['full_name'] = $user['full_name'];
        $_SESSION['login_time'] = time();

        // ایجاد Session Token
        $session_token = bin2hex(random_bytes(32));
        $_SESSION['session_token'] = $session_token;

        // ذخیره در دیتابیس
        $this->db->insert('admin_sessions', [
            'id' => session_id(),
            'admin_id' => $admin_id,
            'ip_address' => $this->getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'expires_at' => date('Y-m-d H:i:s', time() + SESSION_LIFETIME)
        ]);

        $this->admin_id = $admin_id;
        $this->admin_data = $user;
    }

    /**
     * خروج
     */
    public function logout() {
        // حذف Session از دیتابیس
        $this->db->delete('admin_sessions', 'id = :id', [':id' => session_id()]);

        // ثبت لاگ
        if ($this->admin_id) {
            $this->logActivity($this->admin_id, 'logout', 'admin', $this->admin_id);
        }

        // پاک کردن Session
        $_SESSION = [];
        session_destroy();

        return true;
    }

    /**
     * بررسی احراز هویت
     */
    public function isAuthenticated() {
        return isset($_SESSION['admin_id']) && $this->admin_id !== null;
    }

    /**
     * بررسی دسترسی
     */
    public function hasRole($role) {
        if (!$this->isAuthenticated()) {
            return false;
        }

        $roles = [
            ROLE_SUPER_ADMIN => 3,
            ROLE_ADMIN => 2,
            ROLE_MODERATOR => 1
        ];

        $current_level = $roles[$this->admin_data['role']] ?? 0;
        $required_level = $roles[$role] ?? 0;

        return $current_level >= $required_level;
    }

    /**
     * بررسی دسترسی خاص
     */
    public function hasPermission($permission) {
        if (!$this->isAuthenticated()) {
            return false;
        }

        // Super Admin همه دسترسی‌ها را دارد
        if ($this->admin_data['role'] === ROLE_SUPER_ADMIN) {
            return true;
        }

        $permissions = json_decode($this->admin_data['permissions'] ?? '[]', true);
        return in_array($permission, $permissions);
    }

    /**
     * الزام احراز هویت
     */
    public function requireAuth() {
        if (!$this->isAuthenticated()) {
            header('Location: login.php');
            exit;
        }
    }

    /**
     * الزام سطح دسترسی
     */
    public function requireRole($role) {
        $this->requireAuth();

        if (!$this->hasRole($role)) {
            http_response_code(403);
            die('دسترسی غیرمجاز');
        }
    }

    /**
     * دریافت اطلاعات ادمین
     */
    public function getAdmin() {
        return $this->admin_data;
    }

    /**
     * دریافت ID ادمین
     */
    public function getAdminId() {
        return $this->admin_id;
    }

    /**
     * بارگذاری اطلاعات ادمین
     */
    private function loadAdminData() {
        $this->admin_data = $this->db->query("
            SELECT u.*, a.id as admin_id, a.role, a.permissions
            FROM users u
            INNER JOIN admins a ON u.id = a.user_id
            WHERE a.id = :admin_id
            LIMIT 1
        ")
        ->bind(':admin_id', $this->admin_id)
        ->fetch();

        if (!$this->admin_data) {
            $this->logout();
        }
    }

    /**
     * بررسی مسدودیت ورود
     */
    private function isLoginBlocked($email) {
        $key = 'login_attempts_' . md5($email);
        $attempts = $_SESSION[$key] ?? 0;
        $blocked_until = $_SESSION[$key . '_blocked'] ?? 0;

        if ($blocked_until > time()) {
            return true;
        }

        if ($attempts >= MAX_LOGIN_ATTEMPTS) {
            $_SESSION[$key . '_blocked'] = time() + LOGIN_TIMEOUT;
            return true;
        }

        return false;
    }

    /**
     * ثبت تلاش ناموفق
     */
    private function logFailedAttempt($email) {
        $key = 'login_attempts_' . md5($email);
        $_SESSION[$key] = ($_SESSION[$key] ?? 0) + 1;
    }

    /**
     * ثبت فعالیت ادمین
     */
    public function logActivity($admin_id, $action, $entity_type = null, $entity_id = null, $details = []) {
        $this->db->insert('admin_activity_logs', [
            'admin_id' => $admin_id,
            'action' => $action,
            'entity_type' => $entity_type,
            'entity_id' => $entity_id,
            'details' => json_encode($details),
            'ip_address' => $this->getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
    }

    /**
     * دریافت IP کلاینت
     */
    private function getClientIP() {
        $ip = '';
        if (isset($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } elseif (isset($_SERVER['HTTP_X_FORWARDED'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED'];
        } elseif (isset($_SERVER['HTTP_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_FORWARDED_FOR'];
        } elseif (isset($_SERVER['HTTP_FORWARDED'])) {
            $ip = $_SERVER['HTTP_FORWARDED'];
        } elseif (isset($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return $ip;
    }

    /**
     * پاکسازی Session‌های منقضی شده
     */
    public static function cleanExpiredSessions() {
        $db = Database::getInstance();
        $db->delete('admin_sessions', 'expires_at < NOW()');
    }
}
