<?php
define('AYENEH_ADMIN', true);
require_once 'php/autoload.php';

$pageTitle = 'ایجاد اعلان جدید';
$db = Database::getInstance();
$session = Session::getInstance();
$auth = Auth::getInstance();

$errors = [];
$formData = [];

if (isPost()) {
    // دریافت داده‌های فرم
    $formData = [
        'type' => post('type'),
        'target_user_id' => post('target_user_id'),
        'target_plan' => post('target_plan'),
        'target_status' => post('target_status'),
        'target_gender' => post('target_gender'),
        'title' => sanitize(post('title')),
        'body' => sanitize(post('body')),
        'icon' => sanitize(post('icon')),
        'image' => sanitize(post('image')),
        'link' => sanitize(post('link')),
        'send_type' => post('send_type'),
        'scheduled_at' => post('scheduled_at')
    ];

    // اعتبارسنجی
    if (empty($formData['type'])) {
        $errors[] = 'نوع ارسال الزامی است.';
    }

    if ($formData['type'] === 'single' && empty($formData['target_user_id'])) {
        $errors[] = 'برای ارسال تک نفره، انتخاب کاربر الزامی است.';
    }

    if (empty($formData['title'])) {
        $errors[] = 'عنوان اعلان الزامی است.';
    }

    if (empty($formData['body'])) {
        $errors[] = 'متن اعلان الزامی است.';
    }

    if ($formData['send_type'] === 'scheduled' && empty($formData['scheduled_at'])) {
        $errors[] = 'برای ارسال زمان‌بندی شده، تاریخ و زمان الزامی است.';
    }

    if (empty($errors)) {
        try {
            $db->beginTransaction();

            // محاسبه تعداد مخاطبان
            $targetCount = 0;
            $targetFilter = null;

            if ($formData['type'] === 'single') {
                $targetCount = 1;
            } elseif ($formData['type'] === 'group') {
                // ساخت فیلتر گروهی
                $filter = [];
                if ($formData['target_plan']) {
                    $filter['plan'] = $formData['target_plan'];
                }
                if ($formData['target_status']) {
                    $filter['status'] = $formData['target_status'];
                }
                if ($formData['target_gender']) {
                    $filter['gender'] = $formData['target_gender'];
                }

                // شمارش کاربران
                $where = ["u.status != 'deleted'"];
                $params = [];

                if ($formData['target_plan']) {
                    $where[] = "s.plan_type = :plan";
                    $params[':plan'] = $formData['target_plan'];
                }
                if ($formData['target_status']) {
                    $where[] = "u.status = :status";
                    $params[':status'] = $formData['target_status'];
                }
                if ($formData['target_gender']) {
                    $where[] = "u.gender = :gender";
                    $params[':gender'] = $formData['target_gender'];
                }

                $whereClause = implode(' AND ', $where);
                $countQuery = $db->query("
                    SELECT COUNT(DISTINCT u.id)
                    FROM users u
                    LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
                    WHERE {$whereClause}
                ");
                foreach ($params as $key => $value) {
                    $countQuery->bind($key, $value);
                }
                $targetCount = $countQuery->fetchColumn();

                $targetFilter = json_encode($filter);

            } else { // all
                $targetCount = $db->count('users', "status != 'deleted'");
            }

            // ایجاد اعلان
            $notificationData = [
                'type' => $formData['type'],
                'target_user_id' => $formData['target_user_id'] ?: null,
                'target_filter' => $targetFilter,
                'target_count' => $targetCount,
                'title' => $formData['title'],
                'body' => $formData['body'],
                'icon' => $formData['icon'] ?: null,
                'image' => $formData['image'] ?: null,
                'link' => $formData['link'] ?: null,
                'status' => $formData['send_type'] === 'immediate' ? 'pending' : 'scheduled',
                'scheduled_at' => $formData['send_type'] === 'scheduled' ? $formData['scheduled_at'] : null,
                'admin_id' => $auth->getUserId(),
                'created_at' => date('Y-m-d H:i:s')
            ];

            $notificationId = $db->insert('notifications', $notificationData);

            $db->commit();

            $message = $formData['send_type'] === 'immediate'
                ? 'اعلان با موفقیت ایجاد شد و به زودی ارسال می‌شود.'
                : 'اعلان با موفقیت زمان‌بندی شد.';

            $session->setSuccess($message);
            redirect('notifications.php');

        } catch (Exception $e) {
            $db->rollback();
            $errors[] = 'خطا در ایجاد اعلان: ' . $e->getMessage();
            logError('Error creating notification: ' . $e->getMessage());
        }
    }
}

include 'php/includes/header.php';
?>


<div class="row clearfix">
    <div class="col-lg-8 col-md-10 col-sm-12 col-xs-12" style="margin: 0 auto; float: none;">
        <div class="card">
            <div class="header">
                <h2>فرم ایجاد اعلان</h2>
            </div>
            <div class="body">
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <strong>خطاها:</strong>
                        <ul class="m-b-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo h($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <form method="POST" action="" id="notificationForm">
                    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">

                    <h4 class="m-t-0">مخاطبان</h4>

                    <div class="form-group">
                        <label>نوع ارسال <span class="text-danger">*</span></label>
                        <select name="type" id="notificationType" class="form-control" required onchange="toggleTargetFields()">
                            <option value="">انتخاب کنید...</option>
                            <option value="single" <?php echo ($formData['type'] ?? '') === 'single' ? 'selected' : ''; ?>>ارسال به یک کاربر خاص</option>
                            <option value="group" <?php echo ($formData['type'] ?? '') === 'group' ? 'selected' : ''; ?>>ارسال به گروه خاصی از کاربران</option>
                            <option value="all" <?php echo ($formData['type'] ?? '') === 'all' ? 'selected' : ''; ?>>ارسال به همه کاربران</option>
                        </select>
                    </div>

                    <!-- برای ارسال تک نفره -->
                    <div id="singleTarget" style="display: none;">
                        <div class="form-group">
                            <label>انتخاب کاربر</label>
                            <div class="form-line">
                                <input type="number" name="target_user_id" class="form-control"
                                       placeholder="ID کاربر" value="<?php echo h($formData['target_user_id'] ?? ''); ?>">
                            </div>
                            <small class="text-muted">شماره ID کاربر را وارد کنید</small>
                        </div>
                    </div>

                    <!-- برای ارسال گروهی -->
                    <div id="groupTarget" style="display: none;">
                        <div class="alert alert-info">
                            <i class="material-icons">info</i>
                            فیلترهای زیر را انتخاب کنید. کاربرانی که تمام شرایط را داشته باشند، اعلان را دریافت خواهند کرد.
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label>پلن اشتراک</label>
                                    <select name="target_plan" class="form-control">
                                        <option value="">همه</option>
                                        <option value="free_trial">آزمایشی</option>
                                        <option value="freemium">فریمیوم</option>
                                        <option value="plus">پلاس</option>
                                        <option value="premium">پریمیوم</option>
                                    </select>
                                </div>
                            </div>

                            <div class="col-md-4">
                                <div class="form-group">
                                    <label>وضعیت حساب</label>
                                    <select name="target_status" class="form-control">
                                        <option value="">همه</option>
                                        <option value="active">فعال</option>
                                        <option value="suspended">معلق</option>
                                    </select>
                                </div>
                            </div>

                            <div class="col-md-4">
                                <div class="form-group">
                                    <label>جنسیت</label>
                                    <select name="target_gender" class="form-control">
                                        <option value="">همه</option>
                                        <option value="male">مرد</option>
                                        <option value="female">زن</option>
                                        <option value="other">سایر</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>

                    <hr>
                    <h4>محتوای اعلان</h4>

                    <div class="form-group">
                        <label>عنوان <span class="text-danger">*</span></label>
                        <div class="form-line">
                            <input type="text" name="title" class="form-control" required
                                   maxlength="100" value="<?php echo h($formData['title'] ?? ''); ?>"
                                   placeholder="عنوان کوتاه و گویا">
                        </div>
                    </div>

                    <div class="form-group">
                        <label>متن پیام <span class="text-danger">*</span></label>
                        <textarea name="body" class="form-control" rows="4" required
                                  maxlength="500" placeholder="متن اعلان (حداکثر 500 کاراکتر)"><?php echo h($formData['body'] ?? ''); ?></textarea>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>آیکون (URL)</label>
                                <div class="form-line">
                                    <input type="url" name="icon" class="form-control"
                                           placeholder="https://example.com/icon.png"
                                           value="<?php echo h($formData['icon'] ?? ''); ?>">
                                </div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label>تصویر (URL)</label>
                                <div class="form-line">
                                    <input type="url" name="image" class="form-control"
                                           placeholder="https://example.com/image.jpg"
                                           value="<?php echo h($formData['image'] ?? ''); ?>">
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label>لینک (اختیاری)</label>
                        <div class="form-line">
                            <input type="url" name="link" class="form-control"
                                   placeholder="https://example.com/..."
                                   value="<?php echo h($formData['link'] ?? ''); ?>">
                        </div>
                        <small class="text-muted">لینکی که با کلیک روی اعلان باز می‌شود</small>
                    </div>

                    <hr>
                    <h4>زمان‌بندی ارسال</h4>

                    <div class="form-group">
                        <label>زمان ارسال <span class="text-danger">*</span></label>
                        <select name="send_type" id="sendType" class="form-control" required onchange="toggleSchedule()">
                            <option value="immediate">ارسال فوری</option>
                            <option value="scheduled">زمان‌بندی شده</option>
                        </select>
                    </div>

                    <div id="scheduleField" style="display: none;">
                        <div class="form-group">
                            <label>تاریخ و زمان ارسال</label>
                            <div class="form-line">
                                <input type="datetime-local" name="scheduled_at" class="form-control"
                                       value="<?php echo h($formData['scheduled_at'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12">
                            <button type="submit" class="btn-action btn-save waves-effect">
                                <i class="material-icons">send</i>
                                <span>ایجاد و ارسال اعلان</span>
                            </button>

                            <a href="notifications.php" class="btn-action waves-effect">
                                <i class="material-icons">cancel</i>
                                <span>انصراف</span>
                            </a>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
function toggleTargetFields() {
    const type = document.getElementById('notificationType').value;

    document.getElementById('singleTarget').style.display = type === 'single' ? 'block' : 'none';
    document.getElementById('groupTarget').style.display = type === 'group' ? 'block' : 'none';
}

function toggleSchedule() {
    const sendType = document.getElementById('sendType').value;
    document.getElementById('scheduleField').style.display = sendType === 'scheduled' ? 'block' : 'none';
}

// Initialize on page load
window.addEventListener('DOMContentLoaded', function() {
    toggleTargetFields();
    toggleSchedule();
});
</script>

<?php include 'php/includes/footer.php'; ?>
